import Link from 'next/link';
import { XCircleIcon, MenuIcon } from '@heroicons/react/outline';
import { useEffect, useState } from 'react';
import { NAVBAR_ITEMS } from '../utils/pages.constants';
import { getAllPages } from '../utils/pages.data';

const Navbar = () => {
	const [showNavbar, setShowNavbar] = useState(false);
	const [pages, setPages] = useState([]);

	useEffect(() => {
		const fetchPages = async () => {
			const data = await getAllPages();
			const tempPages = data?.map((page) => ({
				title: page.attributes.page_name,
				id: page.id,
			}));
			setPages(tempPages);
		};
		fetchPages();
	}, []);
	const toggleNavbar = () => {
		setShowNavbar(!showNavbar);
	};
	return (
		<nav className='relative'>
			{/* For larger screens */}
			<div className='hidden lg:flex justify-center px-10 pt-6 sm:px-20 text-base text-template-black whitespace-nowrap space-x-2 sm:space-x-4 scrollbar-hide'>
				{NAVBAR_ITEMS.map((navItem, index) => (
					<div className='flex' key={index}>
						<h2 className='pr-4 cursor-pointer transition duration-100 transform hover:scale-125 active:text-red-500'>
							<Link href={navItem.navLink}>{navItem.title}</Link>
						</h2>
						<div className='w-0.5 h-full bg-slate-200'></div>
					</div>
				))}
				{pages?.map((page) => (
					<div key={page.id} className='flex'>
						<h2 className='pr-4 cursor-pointer transition duration-100 transform hover:scale-125 active:text-red-500'>
							<Link href={`/topic/${page.id}`}>{page.title}</Link>
						</h2>
						<div className='w-0.5 h-full bg-slate-200'></div>
					</div>
				))}
				<div className='flex'>
					<h2 className='pr-4 cursor-pointer transition duration-100 transform hover:scale-125 active:text-red-500'>
						<Link href={'/contact'}>Contact</Link>
					</h2>
					<div className='w-0.5 h-full bg-slate-200'></div>
				</div>
			</div>

			{/* For tablets and phones */}
			<section
				className={`right-0 top-0 bottom-0 fixed flex flex-col justify-start w-3/4 ${
					showNavbar ? 'h-full' : 'h-0'
				} bg-white z-10`}
			>
				<header className='lg:hidden w-full h-20 flex justify-end'>
					<button
						className='btn flex justify-center items-center w-20 h-20'
						onClick={toggleNavbar}
					>
						{showNavbar ? (
							<XCircleIcon className='h-6 w-6 text-black overflow-hidden' />
						) : (
							<MenuIcon className='h-6 w-6 text-black' />
						)}
					</button>
				</header>

				<section className={showNavbar ? 'block h-full' : 'hidden'}>
					<h2 className='text-2xl text-center font-bold mb-6'>Menu</h2>
					<div className='overflow-y-scroll flex flex-col items-center gap-1 h-full'>
						{NAVBAR_ITEMS.map((navItem, index) => (
							<Link key={index} href={navItem.navLink}>
								<button className='w-full text-center text-lg py-3'>
									{navItem.title}
								</button>
							</Link>
						))}

						{pages?.map((page) => (
							<Link key={page.id} href={`/topic/${page.id}`}>
								<button
									onClick={toggleNavbar}
									className='w-full text-center text-lg py-3'
								>
									{page.title}
								</button>
							</Link>
						))}
						<div className='flex'>
							<h2 className='pr-4 cursor-pointer transition duration-100 transform hover:scale-125 active:text-red-500'>
								<Link href={'/contact'}>Contact</Link>
							</h2>
							<div className='w-0.5 h-full bg-slate-200'></div>
						</div>
					</div>
				</section>
			</section>
		</nav>
	);
};

export default Navbar;
