"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
exports.isDataUrl = exports.sort = exports.isUrlRequestable = exports.resolveRequests = exports.getExportCode = exports.getModuleCode = exports.getImportCode = exports.getPreRequester = exports.normalizeSourceMap = exports.getModulesPlugins = exports.shouldUseIcssPlugin = exports.shouldUseModulesPlugins = exports.shouldUseURLPlugin = exports.shouldUseImportPlugin = exports.getFilter = exports.requestify = exports.normalizeUrl = void 0;
var _url = require("url");
var _path = _interopRequireDefault(require("path"));
var _loaderUtils3 = require("next/dist/compiled/loader-utils3");
var _postcssModulesValues = _interopRequireDefault(require("next/dist/compiled/postcss-modules-values"));
var _postcssModulesLocalByDefault = _interopRequireDefault(require("next/dist/compiled/postcss-modules-local-by-default"));
var _postcssModulesExtractImports = _interopRequireDefault(require("next/dist/compiled/postcss-modules-extract-imports"));
var _postcssModulesScope = _interopRequireDefault(require("next/dist/compiled/postcss-modules-scope"));
var _camelcase = _interopRequireDefault(require("./camelcase"));
function _interopRequireDefault(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
const whitespace = '[\\x20\\t\\r\\n\\f]';
const unescapeRegExp = new RegExp(`\\\\([\\da-f]{1,6}${whitespace}?|(${whitespace})|.)`, 'ig');
const matchNativeWin32Path = /^[A-Z]:[/\\]|^\\\\/i;
function unescape(str) {
    return str.replace(unescapeRegExp, (_, escaped, escapedWhitespace)=>{
        const high = `0x${escaped}` - 65536;
        /* eslint-disable line-comment-position */ // NaN means non-codepoint
        // Workaround erroneous numeric interpretation of +"0x"
        // eslint-disable-next-line no-self-compare
        return high !== high || escapedWhitespace ? escaped : high < 0 ? String.fromCharCode(high + 65536) : // eslint-disable-next-line no-bitwise
        String.fromCharCode(high >> 10 | 55296, high & 1023 | 56320);
    /* eslint-enable line-comment-position */ });
}
function normalizePath(file) {
    return _path.default.sep === '\\' ? file.replace(/\\/g, '/') : file;
}
function fixedEncodeURIComponent(str) {
    return str.replace(/[!'()*]/g, (c)=>`%${c.charCodeAt(0).toString(16)}`
    );
}
function normalizeUrl(url, isStringValue) {
    let normalizedUrl = url;
    if (isStringValue && /\\(\n|\r\n|\r|\f)/.test(normalizedUrl)) {
        normalizedUrl = normalizedUrl.replace(/\\(\n|\r\n|\r|\f)/g, '');
    }
    if (matchNativeWin32Path.test(url)) {
        try {
            normalizedUrl = decodeURIComponent(normalizedUrl);
        } catch (error) {
        // Ignores invalid and broken URLs and try to resolve them as is
        }
        return normalizedUrl;
    }
    normalizedUrl = unescape(normalizedUrl);
    if (isDataUrl(url)) {
        return fixedEncodeURIComponent(normalizedUrl);
    }
    try {
        normalizedUrl = decodeURI(normalizedUrl);
    } catch (error) {
    // Ignores invalid and broken URLs and try to resolve them as is
    }
    return normalizedUrl;
}
exports.normalizeUrl = normalizeUrl;
function requestify(url, rootContext) {
    if (/^file:/i.test(url)) {
        return (0, _url).fileURLToPath(url);
    }
    if (/^[a-z][a-z0-9+.-]*:/i.test(url)) {
        return url;
    }
    return url.charAt(0) === '/' ? (0, _loaderUtils3).urlToRequest(url, rootContext) : (0, _loaderUtils3).urlToRequest(url);
}
exports.requestify = requestify;
function getFilter(filter, resourcePath) {
    return (...args)=>{
        if (typeof filter === 'function') {
            return filter(...args, resourcePath);
        }
        return true;
    };
}
exports.getFilter = getFilter;
function shouldUseImportPlugin(options) {
    if (options.modules.exportOnlyLocals) {
        return false;
    }
    if (typeof options.import === 'boolean') {
        return options.import;
    }
    return true;
}
exports.shouldUseImportPlugin = shouldUseImportPlugin;
function shouldUseURLPlugin(options) {
    if (options.modules.exportOnlyLocals) {
        return false;
    }
    if (typeof options.url === 'boolean') {
        return options.url;
    }
    return true;
}
exports.shouldUseURLPlugin = shouldUseURLPlugin;
function shouldUseModulesPlugins(options) {
    return options.modules.compileType === 'module';
}
exports.shouldUseModulesPlugins = shouldUseModulesPlugins;
function shouldUseIcssPlugin(options) {
    return options.icss === true || Boolean(options.modules);
}
exports.shouldUseIcssPlugin = shouldUseIcssPlugin;
function getModulesPlugins(options, loaderContext) {
    const { mode , getLocalIdent , localIdentName , localIdentContext , localIdentHashPrefix , localIdentRegExp ,  } = options.modules;
    let plugins = [];
    try {
        plugins = [
            _postcssModulesValues.default,
            (0, _postcssModulesLocalByDefault).default({
                mode
            }),
            (0, _postcssModulesExtractImports).default(),
            (0, _postcssModulesScope).default({
                generateScopedName (exportName) {
                    return getLocalIdent(loaderContext, localIdentName, exportName, {
                        context: localIdentContext,
                        hashPrefix: localIdentHashPrefix,
                        regExp: localIdentRegExp
                    });
                },
                exportGlobals: options.modules.exportGlobals
            }), 
        ];
    } catch (error) {
        loaderContext.emitError(error);
    }
    return plugins;
}
exports.getModulesPlugins = getModulesPlugins;
const IS_NATIVE_WIN32_PATH = /^[a-z]:[/\\]|^\\\\/i;
const ABSOLUTE_SCHEME = /^[a-z0-9+\-.]+:/i;
function getURLType(source) {
    if (source[0] === '/') {
        if (source[1] === '/') {
            return 'scheme-relative';
        }
        return 'path-absolute';
    }
    if (IS_NATIVE_WIN32_PATH.test(source)) {
        return 'path-absolute';
    }
    return ABSOLUTE_SCHEME.test(source) ? 'absolute' : 'path-relative';
}
function normalizeSourceMap(map, resourcePath) {
    let newMap = map;
    // Some loader emit source map as string
    // Strip any JSON XSSI avoidance prefix from the string (as documented in the source maps specification), and then parse the string as JSON.
    if (typeof newMap === 'string') {
        newMap = JSON.parse(newMap);
    }
    delete newMap.file;
    const { sourceRoot  } = newMap;
    delete newMap.sourceRoot;
    if (newMap.sources) {
        // Source maps should use forward slash because it is URLs (https://github.com/mozilla/source-map/issues/91)
        // We should normalize path because previous loaders like `sass-loader` using backslash when generate source map
        newMap.sources = newMap.sources.map((source)=>{
            // Non-standard syntax from `postcss`
            if (source.indexOf('<') === 0) {
                return source;
            }
            const sourceType = getURLType(source);
            // Do no touch `scheme-relative` and `absolute` URLs
            if (sourceType === 'path-relative' || sourceType === 'path-absolute') {
                const absoluteSource = sourceType === 'path-relative' && sourceRoot ? _path.default.resolve(sourceRoot, normalizePath(source)) : normalizePath(source);
                return _path.default.relative(_path.default.dirname(resourcePath), absoluteSource);
            }
            return source;
        });
    }
    return newMap;
}
exports.normalizeSourceMap = normalizeSourceMap;
function getPreRequester({ loaders , loaderIndex  }) {
    const cache = Object.create(null);
    return (number)=>{
        if (cache[number]) {
            return cache[number];
        }
        if (number === false) {
            cache[number] = '';
        } else {
            const loadersRequest = loaders.slice(loaderIndex, loaderIndex + 1 + (typeof number !== 'number' ? 0 : number)).map((x)=>x.request
            ).join('!');
            cache[number] = `-!${loadersRequest}!`;
        }
        return cache[number];
    };
}
exports.getPreRequester = getPreRequester;
function getImportCode(imports, options) {
    let code = '';
    for (const item of imports){
        const { importName , url , icss  } = item;
        if (options.esModule) {
            if (icss && options.modules.namedExport) {
                code += `import ${options.modules.exportOnlyLocals ? '' : `${importName}, `}* as ${importName}_NAMED___ from ${url};\n`;
            } else {
                code += `import ${importName} from ${url};\n`;
            }
        } else {
            code += `var ${importName} = require(${url});\n`;
        }
    }
    return code ? `// Imports\n${code}` : '';
}
exports.getImportCode = getImportCode;
function normalizeSourceMapForRuntime(map, loaderContext) {
    const resultMap = map ? map.toJSON() : null;
    if (resultMap) {
        delete resultMap.file;
        resultMap.sourceRoot = '';
        resultMap.sources = resultMap.sources.map((source)=>{
            // Non-standard syntax from `postcss`
            if (source.indexOf('<') === 0) {
                return source;
            }
            const sourceType = getURLType(source);
            if (sourceType !== 'path-relative') {
                return source;
            }
            const resourceDirname = _path.default.dirname(loaderContext.resourcePath);
            const absoluteSource = _path.default.resolve(resourceDirname, source);
            const contextifyPath = normalizePath(_path.default.relative(loaderContext.rootContext, absoluteSource));
            return `webpack://${contextifyPath}`;
        });
    }
    return JSON.stringify(resultMap);
}
function getModuleCode(result, api, replacements, options, loaderContext) {
    if (options.modules.exportOnlyLocals === true) {
        return '';
    }
    const sourceMapValue = options.sourceMap ? `,${normalizeSourceMapForRuntime(result.map, loaderContext)}` : '';
    let code = JSON.stringify(result.css);
    let beforeCode = `var ___CSS_LOADER_EXPORT___ = ___CSS_LOADER_API_IMPORT___(${options.sourceMap});\n`;
    for (const item of api){
        const { url , media , dedupe  } = item;
        beforeCode += url ? `___CSS_LOADER_EXPORT___.push([module.id, ${JSON.stringify(`@import url(${url});`)}${media ? `, ${JSON.stringify(media)}` : ''}]);\n` : `___CSS_LOADER_EXPORT___.i(${item.importName}${media ? `, ${JSON.stringify(media)}` : dedupe ? ', ""' : ''}${dedupe ? ', true' : ''});\n`;
    }
    for (const item1 of replacements){
        const { replacementName , importName , localName  } = item1;
        if (localName) {
            code = code.replace(new RegExp(replacementName, 'g'), ()=>options.modules.namedExport ? `" + ${importName}_NAMED___[${JSON.stringify((0, _camelcase).default(localName))}] + "` : `" + ${importName}.locals[${JSON.stringify(localName)}] + "`
            );
        } else {
            const { hash , needQuotes  } = item1;
            const getUrlOptions = [].concat(hash ? [
                `hash: ${JSON.stringify(hash)}`
            ] : []).concat(needQuotes ? 'needQuotes: true' : []);
            const preparedOptions = getUrlOptions.length > 0 ? `, { ${getUrlOptions.join(', ')} }` : '';
            beforeCode += `var ${replacementName} = ___CSS_LOADER_GET_URL_IMPORT___(${importName}${preparedOptions});\n`;
            code = code.replace(new RegExp(replacementName, 'g'), ()=>`" + ${replacementName} + "`
            );
        }
    }
    return `${beforeCode}// Module\n___CSS_LOADER_EXPORT___.push([module.id, ${code}, ""${sourceMapValue}]);\n`;
}
exports.getModuleCode = getModuleCode;
function dashesCamelCase(str) {
    return str.replace(/-+(\w)/g, (match, firstLetter)=>firstLetter.toUpperCase()
    );
}
function getExportCode(exports, replacements, options) {
    let code = '// Exports\n';
    let localsCode = '';
    const addExportToLocalsCode = (name, value)=>{
        if (options.modules.namedExport) {
            localsCode += `export const ${(0, _camelcase).default(name)} = ${JSON.stringify(value)};\n`;
        } else {
            if (localsCode) {
                localsCode += `,\n`;
            }
            localsCode += `\t${JSON.stringify(name)}: ${JSON.stringify(value)}`;
        }
    };
    for (const { name: name1 , value: value1  } of exports){
        switch(options.modules.exportLocalsConvention){
            case 'camelCase':
                {
                    addExportToLocalsCode(name1, value1);
                    const modifiedName = (0, _camelcase).default(name1);
                    if (modifiedName !== name1) {
                        addExportToLocalsCode(modifiedName, value1);
                    }
                    break;
                }
            case 'camelCaseOnly':
                {
                    addExportToLocalsCode((0, _camelcase).default(name1), value1);
                    break;
                }
            case 'dashes':
                {
                    addExportToLocalsCode(name1, value1);
                    const modifiedName = dashesCamelCase(name1);
                    if (modifiedName !== name1) {
                        addExportToLocalsCode(modifiedName, value1);
                    }
                    break;
                }
            case 'dashesOnly':
                {
                    addExportToLocalsCode(dashesCamelCase(name1), value1);
                    break;
                }
            case 'asIs':
            default:
                addExportToLocalsCode(name1, value1);
                break;
        }
    }
    for (const item of replacements){
        const { replacementName , localName  } = item;
        if (localName) {
            const { importName  } = item;
            localsCode = localsCode.replace(new RegExp(replacementName, 'g'), ()=>{
                if (options.modules.namedExport) {
                    return `" + ${importName}_NAMED___[${JSON.stringify((0, _camelcase).default(localName))}] + "`;
                } else if (options.modules.exportOnlyLocals) {
                    return `" + ${importName}[${JSON.stringify(localName)}] + "`;
                }
                return `" + ${importName}.locals[${JSON.stringify(localName)}] + "`;
            });
        } else {
            localsCode = localsCode.replace(new RegExp(replacementName, 'g'), ()=>`" + ${replacementName} + "`
            );
        }
    }
    if (options.modules.exportOnlyLocals) {
        code += options.modules.namedExport ? localsCode : `${options.esModule ? 'export default' : 'module.exports ='} {\n${localsCode}\n};\n`;
        return code;
    }
    if (localsCode) {
        code += options.modules.namedExport ? localsCode : `___CSS_LOADER_EXPORT___.locals = {\n${localsCode}\n};\n`;
    }
    code += `${options.esModule ? 'export default' : 'module.exports ='} ___CSS_LOADER_EXPORT___;\n`;
    return code;
}
exports.getExportCode = getExportCode;
async function resolveRequests(resolve, context, possibleRequests) {
    return resolve(context, possibleRequests[0]).then((result)=>{
        return result;
    }).catch((error)=>{
        const [, ...tailPossibleRequests] = possibleRequests;
        if (tailPossibleRequests.length === 0) {
            throw error;
        }
        return resolveRequests(resolve, context, tailPossibleRequests);
    });
}
exports.resolveRequests = resolveRequests;
function isUrlRequestable(url) {
    // Protocol-relative URLs
    if (/^\/\//.test(url)) {
        return false;
    }
    // `file:` protocol
    if (/^file:/i.test(url)) {
        return true;
    }
    // Absolute URLs
    if (/^[a-z][a-z0-9+.-]*:/i.test(url)) {
        return true;
    }
    // `#` URLs
    if (/^#/.test(url)) {
        return false;
    }
    return true;
}
exports.isUrlRequestable = isUrlRequestable;
function sort(a, b) {
    return a.index - b.index;
}
exports.sort = sort;
function isDataUrl(url) {
    if (/^data:/i.test(url)) {
        return true;
    }
    return false;
}
exports.isDataUrl = isDataUrl;

//# sourceMappingURL=utils.js.map